(() => {
    'use strict';

    // カスタム武器データベースアクセス用宣言
    var $customWeapons = null;

    // gamesystemに拡張武器オブジェクトの追加
    const _Game_System_initialize = Game_System.prototype.initialize;
    Game_System.prototype.initialize = function() {
        _Game_System_initialize.call(this);
        this._customWeapons = {};
        this._weaponId = 0;
    };

    // 武器IDの取得処理
    Game_System.prototype.addCustomWeapon = function() {
        this._weaponId += 1;
        return this._weaponId;
    };

    // アイテム入手時の処理
    const _Game_Party_gainItem = Game_Party.prototype.gainItem;
    Game_Party.prototype.gainItem = function(item, amount, includeEquip) {
        if (DataManager.isBaseWeapon(item) && amount > 0) {
            for (let i = 0; i < amount; i++) {
                const newWeapon = this.createCustomWeapon(item);
                $customWeapons[newWeapon.id] = newWeapon;

                _Game_Party_gainItem.call(this, newWeapon, 1, includeEquip);
            }
        }
        else
        {
            _Game_Party_gainItem.call(this, item, amount, includeEquip);
        }
    };

    // パーティ内のキャラからレアリティアップの値を取得
    Game_Party.prototype.getRairityUpRate = function() {
        let rate = 0;
        for (const actor of this.members()){
            rate += actor.traitsRarityUp();
        }
        return rate;
    }

    // カスタム武器の作成処理
    Game_Party.prototype.createCustomWeapon = function(item) {
        // 既存データをコピーしてIDを新規割り振り
        const newWeapon = this.copyData(item);
        const newId = $gameSystem.addCustomWeapon();
        newWeapon.id = newId;

        // 攻撃力HPを基準からばらつかせる
        const rate = Math.floor(Math.random() * 41) + 80;
        newWeapon.atk = Math.floor(newWeapon.baseAtk * rate / 100);
        newWeapon.hp = Math.floor(newWeapon.baseHp * rate / 100);

        // レアリティのセット
        newWeapon.rarity = Math.floor(Math.pow(Math.random(), 2.5) * 101);

        // レアリティアップの効果適用
        const rarityUpRate = this.getRairityUpRate();
        const floorBonus = Math.floor(rarityUpRate * 0.3); 
        const scaleBonus = Math.floor((100 - newWeapon.rarity) * rarityUpRate / 100);
        const bonus = Math.max(floorBonus, scaleBonus);

        // ボーナス値の適用
        newWeapon.rarity = Math.floor(Math.min(100, newWeapon.rarity + bonus));

        // レアリティの値割る１０回分特徴をセット
        const count = Math.max(Math.floor(newWeapon.rarity / 10), 1);
        for(let i = 0; i < count; i++){
            this.setTrait(newWeapon);
        }

        // 特徴の数に応じて名前を変更
        if (count != 1){
            newWeapon.name = newWeapon.name + "＋" + String(count - 1).replace(/[0-9]/g, c =>
                    String.fromCharCode(c.charCodeAt(0) + 0xFEE0)
                );
        }

        // カスタム武器を返す
        return newWeapon;
    }

    // データのコピー処理
    Game_Party.prototype.copyData = function(base) {
        const copy = JSON.parse(JSON.stringify(base));
        return copy;
    }

    // 特徴の値をカスタムして武器にセットする処理
    Game_Party.prototype.setTrait = function(item) {
        // 偏り防止のために毎回シャッフル
        this.traitShuffle();

        // 特徴の抽選処理
        const trait = this.getRandomTrait();

        // 判定用のランク値の取得処理
        const rankValue = (4 + item.rank);

        // 実行値の取得
        const r1 = Math.random();
        const r2 = Math.random();
        const r = (r1 + r2) / 2; 

        // 実行値はランダムに変化させる
        trait.value = trait.baseValue * (Math.floor(r * rankValue) + 1);
        trait.value = Math.floor(trait.value * 100) / 100;

        // 特徴をセット
        item.traits.push(trait);
    }

    // 特徴データからランダムで特徴を取得する処理
    Game_Party.prototype.getRandomTrait = function() {
        const pool = $dataTraits.filter(t => t);
        const total = pool.reduce((s, t) => s + t.rate, 0);
        let r = Math.random() * total;

        for (const t of pool) {
            r -= t.rate;
            if (r <= 0) return this.copyData(t);
        }
        return this.copyData(pool[0]);
    }

    // 特徴データをシャッフルする処理（偏り対策用）
    Game_Party.prototype.traitShuffle = function() {
        for (let i = $dataTraits.length - 1; i > 0; i--) {
            const j = Math.floor(Math.random() * (i + 1));
            [$dataTraits[i], $dataTraits[j]] = [$dataTraits[j], $dataTraits[i]];
        }
    }

    // 装備変更時のアイテム取得処理
    // こっちはデータを追加する必要がないからアイテムの数を減らすだけでいい
    Game_Party.prototype.replaceEquipGainItem = function(item, amount = 1, includeEquip = true) {
        _Game_Party_gainItem.call(this, item, amount, includeEquip);
    };

    // アイテム喪失時の処理
    const _Game_Party_loseItem = Game_Party.prototype.loseItem;
    Game_Party.prototype.loseItem = function(item, amount, includeEquip) {
        if (DataManager.isWeapon(item) && amount > 0) {
            for (let i = 0; i < amount; i++) {
                if (item.id != null) {
                    delete $customWeapons[item.id];
                }
                _Game_Party_loseItem.call(this, item, 1, includeEquip);
            }
        }
        else
        {
            _Game_Party_loseItem.call(this, item, amount, includeEquip);
        }
    };

    // 装備変更時のアイテム喪失処理
    // こっちはデータを削除する必要がないからアイテムの数を減らすだけでいい
    Game_Party.prototype.replaceEquipLoseItem = function(item, amount = 1, includeEquip = true) {
        _Game_Party_loseItem.call(this, item, amount, includeEquip);
    };

    // 所持してる武器情報の取得
    Game_Party.prototype.weapons = function() {
        return Object.keys(this._weapons).map(id => $customWeapons[id]);
    };

    // アクターの装備変更時の処理
    // カスタムデータように装備変更時の取得・喪失処理は専用の関数を使う
    Game_Actor.prototype.tradeItemWithParty = function(newItem, oldItem) {
        if (newItem && !$gameParty.hasItem(newItem)) {
            return false;
        } else {
            $gameParty.replaceEquipGainItem(oldItem, 1);
            $gameParty.replaceEquipLoseItem(newItem, 1);
            return true;
        }
    };

    // 基本の武器かどうかの判定処理
    DataManager.isBaseWeapon = function(item) {
        return item && item.id != null && $dataWeapons[item.id] === item;
    };

    // 拡張した武器かどうかの判定処理
    DataManager.isWeapon = function(item) {
        return item && item.id != null && $customWeapons[item.id] === item;
    };

    // たぶんアイテムの情報取ってくるところ
    Game_Item.prototype.object = function() {
        if (this.isSkill()) {
            return $dataSkills[this._itemId];
        } else if (this.isItem()) {
            return $dataItems[this._itemId];
        } else if (this.isWeapon()) {
            return $customWeapons[this._itemId];
        } else if (this.isArmor()) {
            return $dataArmors[this._itemId];
        } else {
            return null;
        }
    };

    // アクターの装備切り替え
    Game_Actor.prototype.changeEquipById = function(etypeId, itemId) {
        const slotId = etypeId - 1;
        if (this.equipSlots()[slotId] === 1) {
            this.changeEquip(slotId, $customWeapons[itemId]);
        } else {
            this.changeEquip(slotId, $dataArmors[itemId]);
        }
    };

    // 装備している武器以外の武器の削除処理
    Game_Interpreter.prototype.clearAllCustomWeapons = function() {
        const weapons = $gameParty.weapons();
        for (const w of weapons.slice()) {
            $gameParty.loseItem(w, 1, true);
        }
    };

    // 特徴データ読み込みのための処理拡張
    const _DataManager_loadDatabase = DataManager.loadDatabase;
    DataManager.loadDatabase = function() {
        _DataManager_loadDatabase.call(this);

        DataManager.loadDataFile("$dataTraits", "Traits.json");
    };

    // ゲームオブジェクトの作成処理
    const _DataManager_createGameObjects = DataManager.createGameObjects;
    DataManager.createGameObjects = function() {
        _DataManager_createGameObjects.call(this);
        $customWeapons = $gameSystem._customWeapons;
    };

    // ゲームオブジェクトの保存処理
    const _DataManager_extractSaveContents = DataManager.extractSaveContents;
    DataManager.extractSaveContents = function(contents) {
        _DataManager_extractSaveContents.call(this, contents);
        $customWeapons = $gameSystem._customWeapons;
    };

})();